#!/usr/bin/python
#--------------------------------------------------------------------
# Loi binomiale
#
# Marc Lorenzi
# 6 novembre 2013
#--------------------------------------------------------------------

import tkinter as tk
from math import exp, sqrt, pi

def power_dn(n, k):
    p = 1
    for j in range(k): p *= n - j
    return p
    
def factorial(n):
    return power_dn(n, n)
    
def binomial(n, k):
    if k < 0 or k > n: return 0
    return power_dn(n, k) // factorial(k)
    
def binomial_proba(n, p, k):
    return binomial(n, k) * p ** k * (1 - p) ** (n - k)

#--------------------------------------------------------------------

def scale(x, w, xmax):
    return 20 + x * (w - 40) / float(xmax) 


class BinomialFrame(tk.Frame):
    
    
    def __init__(self, parent, width=600, height=400):
        tk.Frame.__init__(self, parent)
        
        self.width = width
        self.height = height
        
        self.canvas = tk.Canvas(self, width=width, height=height, bg='#ffffff')
        self.canvas.grid(column=0, row=2, columnspan=2)
        
        lbln = tk.Label(self, text='n')
        lbln.grid(column=0, row=0, sticky=tk.E+tk.S)
        
        self.scn = tk.Scale(self, orient=tk.HORIZONTAL,
            length=100,
            from_ =1, to_ = 100, 
            resolution = 1, command=lambda _:self.paint())
        self.scn.set(45)
        self.scn.grid(column=1, row=0, sticky=tk.E+tk.W)
            
        lblp = tk.Label(self, text='p')
        lblp.grid(column=0, row=1,  sticky=tk.E+tk.S)
        
        self.scp = tk.Scale(self, orient=tk.HORIZONTAL, 
            length=100,
            from_ =0, to_ = 1, 
            resolution = 0.005, command=lambda _:self.paint())
        self.scp.set(0.2)
        self.scp.grid(column=1, row=1, sticky=tk.E+tk.W)
        
        self.paint()
    
    def plot_axes(self, n, p, proba_max):
        for k in range(11):
            self.canvas.create_line(20, self.height-scale(k, self.height,10),
                self.width-20, self.height-scale(k, self.height,10),fill='black')
            self.canvas.create_text(10, scale(10 - k, self.height, 10),
            text="%.2f"%(proba_max * k/10))
            self.canvas.create_text(self.width - 10, 
                scale(10 - k, self.height, 10),
            text="%.2f" % (k / 10.), fill='red')
        for k in range(n + 1):
            x1 = scale(k, self.width, n + 1)
            x2 = scale(k + 1, self.width, n + 1)
            if k % 5 == 0 or k == n+1:
                self.canvas.create_text((x1 + x2) / 2, self.height - 5, text=str(k),
                justify=tk.CENTER)

    def plot_histogram(self, n, probas, proba_max):
        for k in range(n + 1):
            x1 = scale(k, self.width, n + 1)
            x2 = scale(k + 1, self.width, n + 1) 
            y = probas[k] / proba_max
            self.canvas.create_rectangle(
                x1 + 1, self.height - 20, 
                x2 - 1, self.height - scale(y, self.height, 1), 
                fill="black")
        
    def plot_repartition(self, n, somme_probas):
        for k in range(n + 1):
            x1 = scale(k, self.width, n + 1)
            x2 = scale(k + 1, self.width, n + 1) 
            s = somme_probas[k]
            self.canvas.create_line(
                x1, self.height - scale(s, self.height, 1),
                x2, self.height - scale(s, self.height,1), 
                fill='red', width=5)

    def plot_poisson(self, n, proba_max, m):
        for k in range(n + 1):
            x1 = scale(k, self.width, n + 1)
            x2 = scale(k + 1, self.width, n + 1) 
            y = exp(-m) * m ** k / factorial(k) / proba_max
            self.canvas.create_line(
                x1, self.height - scale(y, self.height, 1),
                x2, self.height - scale(y, self.height,1), 
                fill='blue', width=3)

    def plot_normal(self, n, proba_max, m, sigma):
        for k in range(n + 1):
            x1 = scale(k, self.width, n + 1)
            x2 = scale(k + 1, self.width, n + 1) 
            y1 = exp(-(k-m-0.5) ** 2 / (2 * sigma **2)) / (sigma * sqrt(2*pi))/proba_max
            y2 = exp(-(k+1-m-0.5) ** 2 / (2 * sigma **2)) / (sigma * sqrt(2*pi))/proba_max
            self.canvas.create_line(
                x1, self.height - scale(y1, self.height, 1),
                x2, self.height - scale(y2, self.height,1), 
                fill='green', width=2)

    def paint(self):
        n = self.scn.get()
        p = self.scp.get()
        self.canvas.delete("all")
        self.canvas.create_rectangle(3, 3, self.width, self.height, 
            outline='black')
        probas = [binomial_proba(n, p, k) for k in range(n + 1)]
        proba_max = max(probas)
        somme_probas = (n + 1) * [0]
        for k in range(1, n + 1):
            somme_probas[k] = somme_probas[k - 1] + probas[k] 

        self.plot_axes(n, p, proba_max)
        self.plot_histogram(n, probas, proba_max)
        self.plot_repartition(n, somme_probas)
        self.plot_poisson(n, proba_max, n * p)
        self.plot_normal(n, proba_max, n*p, sqrt(n*p*(1-p)))

def run_gui():
    root = tk.Tk()
    root.resizable(width=False, height=False)
    root.title("Loi binomiale")
    root.focus_force()
    f = BinomialFrame(root)
    f.pack()
    root.mainloop()

if __name__ == '__main__':
    run_gui()
